﻿
#include <stdlib.h>
#include <stdio.h>
#include <time.h> // clock_t
#include <new> // std::bad_alloc
#include <algorithm> // min,max
#include "types.h"
#include "compress.h"

using namespace std;

#define ARRAYLEN(a) (sizeof a / sizeof a[0])

void PrintVersion()
{
	printf("\n");
	printf("Optimal aPLib compressor");
	if (sizeof(void*) > 4) {
		printf(" (x64)\n");
	} else {
		printf(" (x86)\n");
	}
	printf("version 2025.11.01\n");
	printf("by Eugene Larchenko (https://gitlab.com/eugene77)\n");
	printf("\n");
}

void PrintUsage(bool verbose)
{
	printf("Usage:\n");
	printf("  oapack.exe [-b] [-f] <inputfilename> [<outputfilename>]\n");
	if (verbose) {
		printf("  (-b option enables back-to-front compression)\n");
		printf("  (-f option enables faster suboptimal compression)\n");
	}
	printf("\n");
}

bool file_exists(const char* path)
{
	FILE* f = fopen(path, "r");
	if (f != NULL) {
		fclose(f);
	}
	return (f != NULL);
}

void reverse_array(byte* a, int count)
{
	for(int i=0, j=count-1; i < j; i++, j--) {
		auto t = a[i]; a[i] = a[j]; a[j] = t;
	}
}

bool inprogress = false;
void ReportPosition(int pos)
{
	if (inprogress) {
		printf("\r"); // move cursor back
	}
	printf("%d    ", pos);
	inprogress = true;
}
void ReportFinished()
{
	if (inprogress) {
		printf("\r"); // move cursor back
		printf("Finished        \n");
		inprogress = false;
	}
}
void ReportAborted()
{
	if (inprogress) {
		printf("\r\n");
		inprogress = false;
	}
}


int main(int argc, const char* argv[])
{
	int retCode = 10;
	try
	{
		PrintVersion();

		bool isReverseMode = false;
		bool isFastMode = false;

		int a = 1;
		for(; a < argc; a++) {
			if (strcmp(argv[a], "-b") == 0) {
				isReverseMode = true;
			} else if (strcmp(argv[a], "-f") == 0) {
				isFastMode = true;
			} else {
				break;
			}
		}

		// make input file name
		if (a >= argc)
		{
			PrintUsage(true);
			throw 1;
		}
		const char* inputPath = argv[a++];
	
		// make output file name
		bool outputPathSpecified = (a < argc);
		char outputPath[1000];
		const char* s = (a < argc) ? argv[a] : inputPath;
		size_t sl = strlen(s);
		if (sl + 10 > ARRAYLEN(outputPath))
		{
			printf("Path is too long\n");
			throw 2;
		}
		strcpy(outputPath, s);
		if (!outputPathSpecified)
		{
			strcat(outputPath, ".ap");
		}


		data = (byte*)malloc(MAX_INPUT_SIZE + 1);
		if (!data) {
			throw std::bad_alloc();
		}

		FILE* fIn = fopen(inputPath, "rb");
		if (!fIn)
		{
			printf("Error opening input file %s\n", inputPath);
			throw 5;
		}
		else
		{
			size_t fsize = fread(data, 1, MAX_INPUT_SIZE + 1, fIn);
			fclose(fIn);
			if (fsize > MAX_INPUT_SIZE)
			{
				printf("Input file is too large. Max supported file size is %d bytes.\n", MAX_INPUT_SIZE);
				throw 3;
			}
			else if (fsize < 1)
			{
				printf("Input file is too small. Min supported file size is 1 byte.\n");
				throw 3;
			}
			else
			{
				size = (int)fsize;
				data = (byte*)realloc(data, fsize * 2); // double buffer size required for z_function
				if (fsize && !data) {
					throw std::bad_alloc();
				}

				if (file_exists(outputPath))
				{
					// we don't want overwriting file. Don't waste time, abort now.
					printf("Error: output file already exists: %s\n", outputPath);
					throw 6;
				}

				printf("Using back-to-front compression: %s\n", isReverseMode ? "Yes" : "No");
				printf("Compression mode: %s\n", isFastMode ? "fast, suboptimal" : "slow, optimal");
				
				printf("Compressing file: %s\n", inputPath);

				if (isReverseMode)
				{
					reverse_array(data, size);
				}

				int packedSize = Compress(isReverseMode, isFastMode, &ReportPosition);
				ReportFinished();

				if (isReverseMode)
				{
					reverse_array(packedData, packedSize);
				}

				double duration = (double)(endtime - starttime) / CLOCKS_PER_SEC;
				printf("time: %.0f sec\n", duration);

				double ratio = (double)packedSize / max(size, 1);
				if (ratio > 1) {
					ratio = max(ratio, 1.001);
				}
				char* warning = (ratio >= 1) ? "(!)" : "";
				printf("compression: %d / %d = %.3f %s\n", packedSize, size, ratio, warning);

				printf("Writing compressed file: %s\n", outputPath);
				if (file_exists(outputPath))
				{
					// we don't want overwriting file
					printf("Error: output file already exists: %s\n", outputPath);
					throw 6;
				}
				else
				{
					FILE* fOut = fopen(outputPath, "wb");
					if (!fOut)
					{
						printf("Error writing output file\n");
						throw 6;
					}
					else
					{
						size_t written = fwrite(packedData, 1, packedSize, fOut);
						fclose(fOut);
						if (written != packedSize)
						{
							// delete incomplete compressed file
							remove(outputPath);
							printf("Error writing output file\n");
							throw 6;
						}

						printf("All OK\n");
						retCode = 0;
					}
				}
			}
		}
	}
	catch (int code)
	{
		ReportAborted();
		retCode = code;
	}
	catch (std::bad_alloc& e)
	{
		ReportAborted();
		printf("\n");
		printf("Error allocating memory (%s)\n", e.what());
		retCode = 8;
	}
	catch (std::exception& e)
	{
		ReportAborted();
		printf("\n");
		printf("Error: %s\n", e.what());
		retCode = 9;
	}

	//printf("\n");
	return retCode;
};

